var fs = require('fs');

var _ = require('lodash');
var async = require('async');
var moment = require('moment-timezone');
var Excel = require('exceljs');

var configs = require('./../../configs');
var api = require('./../api');
var common = require('./common');

var Api = api.family;
var ApiComments = api.comment;

// var E6_STATUS = "586fd7f450b7a9f6081344e4";
// var E7_STATUS = "58875ff79db991a04d58bd35";
// var E1_STATUS = "578d47031abaec3e8ac50c98";


module.exports.render = common.render;

module.exports.upgrade = function (req, res, next) {
    var context = {err: true, response: {}, body: {}};
    _.set(req, [configs.server.name, 'data'].join('.'), context);
    if (_.indexOf(req.session.rights, 'COMMERCIAL_ASSIGN_FAMILY') === -1)
        req.body = _.omit(req.body, 'user');
    var userId = _.get(req, 'session.user._id', undefined);
    if (req.body.newStatus)
        req.body.status = req.body.newStatus;
    else return next();

    var params = {id: req.params.id, body: req.body, userId: userId};
    Api.update(params, req, function (err, response, body) {
        context = {err: err, response: response, body: body};
        _.set(req, [configs.server.name, 'data'].join('.'), context);
        return next();
    });
};

module.exports.getList = function (req, res, next) {
    Api.getList(req.query.params, req.query.search, req.query.options, req, function (err, response, body) {
        if (parseInt(req.query.childrenList)) {
            body = processChildList(body, parseInt(req.query.isArchive));
        }
        var data = {err: err, response: response, body: body};
        _.set(req, [configs.server.name, 'data'].join('.'), data);
        return next();
    });
};

module.exports.getOne = function (req, res, next) {
    Api.getFamily(req.params.id, req, function (err, response, body) {
        var data = {err: err, response: response, body: body};
        _.set(req, [configs.server.name, 'data'].join('.'), data);
        return next();
    });
};

//todo: in the api
var processChildList = function (familyList, isArchive) {
    var childList = [];
    familyList.forEach(function (family) {
        if (!family.children || !family.children.length) return;
        family.children.forEach(function (child) {
            if (isOutOfDate(child, isArchive)) return;
            var childLine = _.clone(family);
            childLine.child = child;
            delete (childLine.children);
            childList.push(childLine)
        })
    })
    return childList;
}

var isOutOfDate = function (child, isArchive) {
    var birthdate = new Date(child.birthdate);
    return !isArchive ? birthdate.getFullYear() <= ((new Date()).getFullYear() - 4) : birthdate.getFullYear() > ((new Date()).getFullYear() - 4)
}

module.exports.getCount = function (req, res, next) {
    Api.getCount(req.query.params, req.query.search, req, function (err, response, body) {
        var data = {err: err, response: response, body: body};
        _.set(req, [configs.server.name, 'data'].join('.'), data);
        return next();
    });
};

module.exports.getFlags = function (req, res, next) {
    Api.getFlags(req.query.type, req, function (err, response, body) {
        var data = {err: err, response: response, body: body};
        _.set(req, [configs.server.name, 'data'].join('.'), data);
        return next();
    });
};

module.exports.getFamilyComments = function (req, res, next) {
    ApiComments.getByRef(req.params.id, req, function (err, response, body) {
        var data = {err: err, response: response, body: body};
        _.set(req, [configs.server.name, 'data'].join('.'), data);
        return next();
    });
};

var flags = {
    "Nouvelles inscriptions": ['57922c3191033dbcd444c8e1'],
    "En cours de traitement": [],
    "Sans Potentiel - Pas d’appel maplaceencrèche": ['578d47031abaec3e8ac50c9b'],
    "Liste attente entrée en direct": [
        '586fd7f450b7a9f6081344e4', '578d47031abaec3e8ac50c99', '578d47031abaec3e8ac50c9a',
        '578d47031abaec3e8ac50c9d', '58c7b411ca61ddc2fbde2093', '578d47031abaec3e8ac50c9e',
        '58c7b411ca61ddc2fbde2094', '578d47031abaec3e8ac50c9e', '58c7b411ca61ddc2fbde2094',
        '578d47031abaec3e8ac50ca8', '58c7b411ca61ddc2fbde2095'
    ],
    "Contrats en direct": ['58875ff79db991a04d58bd35'],
    "Contrats entreprise": ['578d47031abaec3e8ac50ca0', '578d47031abaec3e8ac50ca9'],
    "Abandons Familles": [
        '578d47031abaec3e8ac50c98', '578d47031abaec3e8ac50ca1', '5828a4cce51ae57725a9abd6',
        '578d47031abaec3e8ac50ca5', '578d47031abaec3e8ac50c9f', '5828a4cce51ae57725a9abd7'
    ]
}

var getFlagFromStatus = function (status) {
    var flag, defaultFlag;
    _.forEach(_.keys(flags), function (f) {
        if (!flag) {
            if (_.isEmpty(flags[f])) defaultFlag = f;
            if (flags[f].indexOf(status) > -1) flag = f;
        }
    })
    if (!flag) flag = defaultFlag;
    return flag;
}

var shortDays = ['Di', 'Lu', 'Ma', 'Me', 'Je', 'Ve', 'Sa'];

module.exports.export = function (req, res, next) {
    res.header("Access-Control-Allow-Origin", "*");
    res.header("Access-Control-Allow-Headers", "Origin, X-Requested-With, Content-Type, Accept");

    var families = [];
    var comments = [];
    var histories = [];
    var companies = [];
    var familyStatutes = [];

    var o = {sort: "created", order: "DESC", populate: true};
    Api.getList(null, null, JSON.stringify(o), req, function (err, response, body) {
        if (err || !body || !_.isArray(body) || body.length < 1) return next();
        families = body;

        var familyIds = [];
        var companyIds = [];
        _.forEach(families, function (f) {
            familyIds.push(f._id);
            _.forEach(f.parents, function (p) {
                var cref = _.get(p, ['company', 'reference']);
                if (cref) companyIds.push(cref);
            })
        })

        async.parallel([
            function (cb) {
                if (familyIds.length < 1) return cb();
                return api.familyHistory.getList(familyIds, 'status', null, req, function (err, response, body) {
                    if (err || !body || !_.isArray(body)) return cb(err);
                    histories = body || [];
                    return cb();
                })
            },
            function (cb) {
                if (familyIds.length < 1) return cb();
                return api.comment.getByRef(familyIds, req, function (err, response, body) {
                    if (err || !body || !_.isArray(body)) return cb(err);
                    comments = body || [];
                    return cb();
                })

            },
            function (cb) {
                if (companyIds.length < 1) return cb();
                var p = {ids: companyIds};
                return api.company.getList(JSON.stringify(p), null, null, req, function (err, response, body) {
                    if (err || !body || !_.isArray(body)) return cb(err);
                    companies = body || [];
                    return cb();
                })
            }
        ], function (err, results) {
            var workbook = new Excel.Workbook();
            var worksheet = workbook.addWorksheet('inscriptions');
            worksheet.columns = [
                {header: 'Date d\'inscription', key: 'created', width: 16},
                {header: 'Civilité (parent 1)', key: 'gender1', width: 5},
                {header: 'Nom (parent 1)', key: 'lastname1', width: 20},
                {header: 'Prénom (parent 1)', key: 'firstname1', width: 20},
                {header: 'Email (parent 1)', key: 'email1', width: 30},
                {header: 'Ville (parent 1)', key: 'address.city1', width: 25},
                {header: 'Code postal (parent 1)', key: 'address.zipCode1', width: 11},
                {header: 'Adresse (parent 1)', key: 'address.street1', width: 30},
                {header: 'Téléphone (parent 1)', key: 'cellphone1', width: 15},
                {header: 'Statut professionnel (parent 1)', key: 'status1', width: 30},
                {header: 'Entreprise (parent 1)', key: 'company1', width: 30},
                {header: 'Civilité (parent 2)', key: 'gender2', width: 5},
                {header: 'Nom (parent 2)', key: 'lastname2', width: 20},
                {header: 'Prénom (parent 2)', key: 'firstname2', width: 20},
                {header: 'Email (parent 2)', key: 'email2', width: 30},
                {header: 'Ville (parent 2)', key: 'address.city2', width: 25},
                {header: 'Code postal (parent 2)', key: 'address.zipCode2', width: 11},
                {header: 'Adresse (parent 2)', key: 'address.street2', width: 30},
                {header: 'Téléphone (parent 2)', key: 'cellphone2', width: 15},
                {header: 'Statut professionnel (parent 2)', key: 'status2', width: 30},
                {header: 'Entreprise (parent 2)', key: 'company2', width: 30},
                {header: 'Revenus', key: 'earnings', width: 7},
                {header: 'Source', key: 'source', width: 30},
                {header: 'Note d\'inscription', key: 'note', width: 50},
                {header: 'Prénom (enfant 1)', key: 'child_firstname1', width: 20},
                {header: 'DDN (enfant 1)', key: 'child_birthdate1', width: 16},
                {header: 'Date d\'entrée (enfant 1)', key: 'child_startdate1', width: 16},
                {header: 'Rythme (enfant 1)', key: 'child_rythm1', width: 16},
                {header: 'Jours (enfant 1)', key: 'child_days1', width: 16},
                {header: 'Prénom (enfant 2)', key: 'child_firstname2', width: 20},
                {header: 'DDN (enfant 2)', key: 'child_birthdate2', width: 16},
                {header: 'Date d\'entrée (enfant 2)', key: 'child_startdate2', width: 16},
                {header: 'Rythme (enfant 2)', key: 'child_rythm2', width: 16},
                {header: 'Jours (enfant 2)', key: 'child_days2', width: 16},
                {header: 'Prénom (enfant 3)', key: 'child_firstname3', width: 20},
                {header: 'DDN (enfant 3)', key: 'child_birthdate3', width: 16},
                {header: 'Date d\'entrée (enfant 3)', key: 'child_startdate3', width: 16},
                {header: 'Rythme (enfant 3)', key: 'child_rythm3', width: 16},
                {header: 'Jours (enfant 3)', key: 'child_days3', width: 16},
                {header: 'Statut actuel (FamilyHub)', key: 'status_fh', width: 50},
                {header: 'Statut actuel (mpec)', key: 'status', width: 50},
                {header: 'Historique des statuts (mpec)', key: 'statutes', width: 50},
                {header: 'Commentaires', key: 'comments', width: 50},
                {header: 'Chargé d\'affaire', key: 'user', width: 20},
                {header: 'État', key: 'flag', width: 10},
                {header: 'Structure', key: 'creche', width: 10}
            ]

            async.eachSeries(families, function (family, cb) {
                var earnings = 0;
                var row = {};
                if (family.created) row.created = moment.tz(family.created, 'Europe/Paris').format('DD/MM/YYYY HH:mm:ss');
                _.forEach(family.parents, function (parent, ip) {
                    if (ip >= 2) return;
                    var civility = _.get(parent, 'gender');
                    if (civility && civility === 'f') row['gender' + (ip + 1)] = 'Mme';
                    if (civility && civility === 'm') row['gender' + (ip + 1)] = 'M';
                    var keys = ['lastname', 'firstname', 'email', 'address.city', 'address.zipCode', 'address.street', 'cellphone', 'status'];
                    _.forEach(keys, function (key) {
                        var val = _.get(parent, key);
                        if (val) row[key + (ip + 1)] = val;
                    })
                    var company;
                    var cid = _.get(parent, ['company', 'reference']);
                    if (cid) {
                        var c = _.find(companies, {_id: cid});
                        company = _.get(c, 'name');
                        if (!company) company = _.get(c, 'denomination', '')
                    } else company = _.get(parent, ['company', 'name'], '');
                    row['company' + +(ip + 1)] = company;
                    earnings += _.get(parent, 'earning', 0);
                })
                row.earnings = earnings;
                var keys = ['source', 'note', 'flag'];
                _.forEach(keys, function (key) {
                    var val = _.get(family, key);
                    if (val) row[key] = val;
                })
                _.forEach(family.children, function (child, ic) {
                    if (ic >= 3) return;
                    var keys = ['firstname', 'rythm'];
                    _.forEach(keys, function (key) {
                        var val = _.get(child, key);
                        if (val) row['child_' + key + (ic + 1)] = val;
                    })
                    var keys = ['birthdate', 'startdate'];
                    _.forEach(keys, function (key) {
                        var val = _.get(child, key);
                        if (val) row['child_' + key + (ic + 1)] = moment.tz(val, 'Europe/Paris').format('DD/MM/YYYY');
                    })
                    var days = _.get(child, 'days', []);
                    _.forEach(days, function (v, k) {
                        days[k] = shortDays[v];
                    })
                    days = days.join(', ');
                    if (days) row['child_days' + (ic + 1)] = days;
                })

                var history = _.filter(histories, {type: 'status', family: family._id});
                row.statutes = '';
                _.forEach(history, function (h) {
                    var status = _.get(h, ['reference', 'label']);
                    var statusId = _.get(h, ['reference', '_id']);
                    if (status) {
                        if (family.status && statusId && family.status === statusId) row.status = status;
                        row.statutes += '# ';
                        row.statutes += moment.tz(h.created, 'Europe/Paris').format('DD/MM/YYYY HH:mm:ss');
                        row.statutes += ' # ';
                        row.statutes += status;
                        row.statutes += '\n';
                    }
                })
                if (family.status) {
                    var status_fh = getFlagFromStatus(family.status);
                    if (status_fh) row.status_fh = status_fh;
                }

                var familyComments = _.filter(comments, {type: 'family', reference: family._id});
                row.comments = '';
                _.forEach(familyComments, function (comment) {
                    row.comments += '# ';
                    row.comments += moment.tz(comment.created, 'Europe/Paris').format('DD/MM/YYYY HH:mm:ss');
                    row.comments += ' #\n';
                    row.comments += comment.content;
                    row.comments += '\n';
                })
                if (family.user && family.user._id) {
                    var name = '';
                    if (family.user.firstname) name += family.user.firstname[0].toUpperCase() + '. ';
                    if (family.user.lastname) name += family.user.lastname.toUpperCase();
                    if (name !== '') row.user = name;
                }
                if (family.note) row.note = family.note;
                var creche = _.get(family, ['meta', 'creche']);
                if (creche) row.creche = creche;
                worksheet.addRow(row);
                return cb();
            }, function (err, cb) {
                var format = {name: 'csv', mime: 'text/csv'};
                if (_.endsWith(req.path, '.xlsx')) format = {
                    name: 'xlsx',
                    mime: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
                };

                console.log(format);

                res.setHeader('Content-Type', format.mime);
                var filename = moment().format('YYYY-MM-DD_HH-mm-ss') + '_inscriptions.' + format.name;
                res.setHeader('Content-Disposition', 'attachment; filename=' + filename);
                if (format.name === "xlsx") {
                    return workbook.xlsx.write(res);
                } else {
                    return workbook.csv.write(res);
                }
            })
        })
    })
}
